import numpy as np

def calculate_features(r_k_t_groundtruth, r_k_t_reconstructed, window_size):
    """
    Calculate ARP and CFO features for each user using a sliding window approach.

    Args:
        r_k_t_reconstructed (np.ndarray): Reconstructed signals, shape (K, T, L)
        window_size (int): Size of the sliding window.

    Returns:
        features: np.ndarray, shape (K, N, 2) where N = T - window_size + 1,
                  with sliding window averages
    """
    K, T, L = r_k_t_reconstructed.shape

    # Ground truth ARP: 
    instant_arp_groundtruth = np.linalg.norm(r_k_t_groundtruth, axis=2)**2  # shape (K, T)

    # Instant ARP: squared norm along last axis
    instant_arp = np.linalg.norm(r_k_t_reconstructed, axis=2)**2  # shape (K, T)

    # Instant CFO
    d_k_t_l = r_k_t_reconstructed[:, :, 1:] / r_k_t_reconstructed[:, :, :-1]  # shape (K, T, L-1)
    dk_t_sum = np.sum(d_k_t_l, axis=2) / (L - 1)  # shape (K, T)
    f_hat_k_t = np.angle(dk_t_sum) / (2 * np.pi)
    instant_cfo = np.real(f_hat_k_t)  # shape (K, T)

    if window_size > T:
        raise ValueError(f"Window size ({window_size}) cannot be larger than total time slots ({T})")

    N = T - window_size + 1

    # Use sliding window with np.lib.stride_tricks.sliding_window_view
    arp_windows_groundtruth = np.lib.stride_tricks.sliding_window_view(instant_arp_groundtruth, window_shape=window_size, axis=1)  # shape (K, N, window_size)
    arp_windows = np.lib.stride_tricks.sliding_window_view(instant_arp, window_shape=window_size, axis=1)  # shape (K, N, window_size)
    cfo_windows = np.lib.stride_tricks.sliding_window_view(instant_cfo, window_shape=window_size, axis=1)  # shape (K, N, window_size)

    arp_windows_groundtruth = np.mean(arp_windows_groundtruth, axis=2)  # shape (K, N)

    features = np.empty((K, N, 2), dtype=np.float64)
    features[..., 0] = np.mean(arp_windows, axis=2)
    features[..., 1] = np.mean(cfo_windows, axis=2)

    return arp_windows_groundtruth, features