import numpy as np

def generate_zc_pilot_matrix(L, u=1):
    """
    Generate an L x L Zadoff-Chu pilot matrix where each column is a cyclic shift
    of the base ZC sequence by the column index.
    
    The base ZC sequence is defined as:
    c(n) = exp(-j * pi * u * n * (n + 1) / L) for n = 0, ..., L-1
    
    Each column is normalized to have unit energy (sum of squared magnitudes = 1),
    matching the normalization of the unitary DFT matrix.
    
    Args:
        L (int): Sequence length (number of subcarriers).
        u (int, optional): ZC root index. Defaults to 1.
    
    Returns:
        np.ndarray: ZC pilot matrix of shape (L, L), where rows correspond to
                    subcarrier indices and columns to different sequences/shifts.
    """
    if L <= 0:
        raise ValueError("L must be a positive integer.")
    
    n = np.arange(L)
    # Generate base ZC sequence (unnormalized, unit magnitude)
    base = np.exp(-1j * np.pi * u * n * (n + 1) / L)
    
    # Normalize base to have energy L (for unit norm after /sqrt(L))
    # Since |base[n]| = 1, sum |base|^2 = L already
    base /= np.sqrt(L)  # Now each shifted column will have norm 1
    
    # Generate shift indices: indices[n, k] = (n + k) % L
    shifts = (np.arange(L)[:, np.newaxis] + np.arange(L)[np.newaxis, :]) % L

#     shifts = [
#   [ (0+0)%4=0, (0+1)%4=1, (0+2)%4=2, (0+3)%4=3 ],  # Row 0
#   [ (1+0)%4=1, (1+1)%4=2, (1+2)%4=3, (1+3)%4=0 ],  # Row 1
#   [ (2+0)%4=2, (2+1)%4=3, (2+2)%4=0, (2+3)%4=1 ],  # Row 2
#   [ (3+0)%4=3, (3+1)%4=0, (3+2)%4=1, (3+3)%4=2 ]   # Row 3
# ]
    
    # Construct matrix using advanced indexing
    zc_matrix = base[shifts]
    
    return zc_matrix